@echo off

REM =========================================================================
REM Parameters:
REM  version - RAD Studio version (5.0 to 23.0, default: 23.0)
REM  debug   - Build in debug mode (optional), default is Release mode
REM
REM Examples:
REM  buildDemo-Win32           	- Build using RAD Studio 23.0 in Release mode
REM  buildDemo-Win32 debug     	- Build using RAD Studio 23.0 in Debug mode
REM  buildDemo-Win32 22.0      	- Build using RAD Studio 22.0 in Release mode
REM  buildDemo-Win32 22.0 debug - Build using RAD Studio 21.0 in Debug mode
REM =========================================================================

echo Usage: buildDemo-Win32 [version] [debug]
echo version is RAD Studio version from 5.0 to 23.0
echo Exempting debug to compiling in Release mode
echo.

setlocal EnableDelayedExpansion

REM Default version is 23.0
set "VERSION=23.0"
set "DEBUG_MODE=0"

REM Parse command line arguments
:parse_args
if "%~1"=="" goto end_parse_args
if /i "%~1"=="debug" (
    set "DEBUG_MODE=1"
) else (
    set "VERSION=%~1"
)
shift
goto parse_args
:end_parse_args

REM Determine registry path based on version
set "REG_PATH=HKLM\SOFTWARE\WOW6432Node\Embarcadero\BDS\"
if "%VERSION%"=="5.0" set "REG_PATH=HKLM\SOFTWARE\WOW6432Node\Borland\BDS\"
for %%i in (6.0 7.0) do if "%VERSION%"=="%%i" set "REG_PATH=HKLM\SOFTWARE\WOW6432Node\CodeGear\BDS\"

REM Get BDS root directory from registry
for /f "tokens=2* delims= " %%i in ('reg query "%REG_PATH%%VERSION%" /v RootDir 2^>nul') DO @set "BDS=%%j"
if defined BDS set "BDS=%BDS:~0,-1%"
if not defined BDS goto noBDS

REM Set build configuration based on debug mode
set "BUILD_CONFIG=Release"
set "DEBUG_FLAGS=-$D- -$L- -$Y- -$C- -$O+ -$R- -$Q- -$I-"
if "%DEBUG_MODE%"=="1" (
    set "BUILD_CONFIG=Debug"
    set "DEBUG_FLAGS=-$D+ -$L+ -$Y+ -$C+ -$O- -$R+ -$Q+ -$I+"
)

set LOGFILE=build-Win32-%BUILD_CONFIG%.log

REM First, delete the old log file if it exists
if exist "%LOGFILE%" del /f /q "%LOGFILE%" 2>nul

REM Create log file
echo Delphi Compilation Log - %DATE% %TIME% > "%LOGFILE%" 2>nul
if not exist "%LOGFILE%" (
    echo Warning: Unable to create log file, will only output to console
    set "LOGFILE="
) else (
    echo. >> "%LOGFILE%" 2>nul
)

echo Searching for Delphi project files (.dpr) ...
echo Building RAD Studio %VERSION% Win32 %BUILD_CONFIG% ...
echo.

set COUNT=0
set FAIL_COUNT=0

for /r "%~dp0" %%f in (*.dpr) do (
    set "FILEPATH=%%f"
    set "FILENAME=%%~nxf"
    REM Ensure the file is actually a .dpr file, not a .dproj file
    if /i "!FILENAME:~-4!"==".dpr" (
        echo Compiling: !FILEPATH!
        
        REM First run the compilation and capture all output to a temporary file
        "%BDS%\bin\dcc32" %DEBUG_FLAGS% -Q -B --codepage:65001 "!FILEPATH!" -U..\..\Lib -NU"%TEMP%" > "%TEMP%\delphi_output.tmp" 2>&1
        set COMPILE_RESULT=!ERRORLEVEL!
        
        REM Display compilation output to the console
        type "%TEMP%\delphi_output.tmp"
        
        REM Check for Fatal errors
        findstr /C:"Fatal:" "%TEMP%\delphi_output.tmp" >nul 2>&1
        if !ERRORLEVEL! equ 0 (
            echo Compilation failed: !FILENAME!
            if defined LOGFILE (
                echo Compilation failed: !FILEPATH! >> "%LOGFILE%" 2>nul
                
                REM Extract error information and log it
                for /f "delims=" %%e in ('type "%TEMP%\delphi_output.tmp" ^| findstr "Fatal:"') do (
                    echo     %%e >> "%LOGFILE%" 2>nul
                )
                echo. >> "%LOGFILE%" 2>nul
            )
            set /a FAIL_COUNT+=1
        ) else if !COMPILE_RESULT! neq 0 (
            echo Compilation failed: !FILENAME!
            if defined LOGFILE (
                echo Compilation failed: !FILEPATH! >> "%LOGFILE%" 2>nul
                echo     Exit code: !COMPILE_RESULT! >> "%LOGFILE%" 2>nul
                echo. >> "%LOGFILE%" 2>nul
            )
            set /a FAIL_COUNT+=1
        ) else (
            echo Compilation successful: !FILENAME!
        )
        
        REM Clear temporary file
        del "%TEMP%\delphi_output.tmp" > nul 2>&1
        echo(
        set /a COUNT+=1
    )
)

REM Calculate success count
set /a SUCCESS_COUNT=COUNT-FAIL_COUNT

echo Compilation Statistics:
echo Total .dpr files compiled: !COUNT!
echo Successful: !SUCCESS_COUNT!
echo Failed: !FAIL_COUNT!

if defined LOGFILE (
    if !FAIL_COUNT! gtr 0 (
        echo Failed projects have been logged to: %LOGFILE%
    ) else (
        echo All projects compiled successfully
        del "%LOGFILE%" > nul 2>&1
    )
)

goto end

:noBDS
echo Cannot find RAD Studio %VERSION% installation
echo Please make sure RAD Studio is installed or specify a different version

:end
timeout /t 3
